#!/usr/bin/env python3
# -*- coding: utf-8 -*-

#######################################################################
# (C) Klet Observatory                  Author: Michaela Honkova
#     http://www.klet.org/                             2.11.2022
#######################################################################
#
# Requirements: Python 3, PyMySQL, sesitek.txt
#
# The script parses .XML files in its folder for object designations
# and counts number of their observations, then connects MySQL database
# (pymysql.connect) to look info from mpcorb table, assigns tags 
# for the objects and writes puc.txt files with statistics.
#
#######################################################################

# MySQL login 
host = '10.246.1.110'
user = 'klet'
password = 'klet'
database = 'klet'

# new class: asteroid
import sys
class Asteroid:
   'Common base class for all minor planet bodies'
   objectCount = 0

   def __init__(self, packDesn, unpackDesn, NEOCPDesn, NObs, Found, Comet, Tags):
      self.packDesn = packDesn
      self.unpackDesn = unpackDesn
      self.NEOCPDesn = NEOCPDesn
      self.NObs = NObs
      self.Found = Found
      self.Comet = Comet
      self.Tags = Tags
      Asteroid.objectCount += 1
   
   def displayCount(self):
     print("Total Objects %d" % Asteroid.objectCount)

   def displayAsteroid(self):
      print( 'Object: ', self.packDesn.ljust(10), self.unpackDesn.ljust(12), self.NEOCPDesn.ljust(10), str(self.NObs).ljust(5), str(self.Comet).ljust(7), self.Tags)

   # 2008 AB1 -> K08A01B
   # https://www.minorplanetcenter.net/iau/info/PackedDes.html
   # Transforms a designation to packed format.       
   def pack(self): 
       self.unpackDesn = self.unpackDesn.strip()   
       # check for faulty data
       if ('C/C' in self.unpackDesn) or ('C/P/' in self.unpackDesn):
           print('*ERROR* Faulty designation '+str(self.unpackDesn)+' found in the data!')
           input('Press enter to exit the script.')
           sys.exit()          
       # survey asteroid
       if (len(self.unpackDesn) == 8) and (self.unpackDesn[6] == '-'):
           frag = self.unpackDesn[-3:]
           if frag == 'P-L':
               self.packDesn = 'PLS' +self.unpackDesn[:4] # 2040 P-L  = PLS2040
           if frag == 'T-1':
               self.packDesn = 'T1S' +self.unpackDesn[:4] # 3138 T-1  = T1S3138
           if frag == 'T-2':
               self.packDesn = 'T2S' +self.unpackDesn[:4] # 1010 T-2  = T2S1010
           if frag == 'T-3':
               self.packDesn = 'T3S' +self.unpackDesn[:4] # 4101 T-3  = T3S4101
           return                                 
       # definitive number of asteroid
       if self.unpackDesn.isdigit():                      # 123456 -> C3456
           if len(self.unpackDesn)<6:
               self.packDesn = self.unpackDesn.zfill(5)
               return
           else:
               num = int(self.unpackDesn[:2])            # 10=A, 11=B, ...
               if num>35:                                # 36=a, 37=b, ...
                   num += 6
               self.packDesn = self.unpackDesn.replace(self.unpackDesn[:2], chr(num+ord('A')-10), 1)
               return
       # definitive number of comet    
       if (self.unpackDesn[-1:] == 'P') or (self.unpackDesn[-1:] == 'D'):
           frag = self.unpackDesn[:-1]    
           if frag.isdigit():                            # 59P -> 0059P
               self.packDesn = self.unpackDesn.zfill(5)
               self.Comet = True
               return
       # NEOCP: mix of letters and numbers without space
       if self.unpackDesn.find(' ') == -1:
           self.NEOCPDesn = self.unpackDesn
           self.unpackDesn = ''
           return          
       # provisore designation of comet
       if self.unpackDesn.find('/') != -1:               # C/2033 L89-C = CK33L89c 
           self.Comet = True
           self.packDesn = self.unpackDesn[0]            # C, P, D, X
           comet = self.unpackDesn[2:]  
           self.packDesn = self.packDesn +chr(int(comet[:2])+ord('A')-10) +comet[2:4] +comet[5]; # 1998 S -> J98S   
           num = comet[6:]                               # 89-C
           # cometary fragment
           frag = ''
           if not comet[6:].isdigit(): 
               frag = comet[-1:]                         # C
               num = num[:-2]                            # 89
           # comets 
           if int(num) < 100:                            # 00 to 99 -> 00 to 99
               self.packDesn = self.packDesn +num.zfill(2)       
           else:                                         # 162 -> G2
               num = int(num[:2])            
               if num>35:                    
                   num += 6  # ...XYZ[\]^_`abc...
               self.packDesn = self.packDesn + chr(num+ord('A')-10) +comet[8]          
           # full comet or fragment
           if frag == '':
               self.packDesn = self.packDesn +'0' 
           else:
               self.packDesn = self.packDesn +frag.swapcase()                     
           return             
       # provisore designation of asteroid               # 1998 SS162 -> J98SG2S
       else:                                             # 1998 S -> J98S                                                   
           self.packDesn = chr(int(self.unpackDesn[:2])+ord('A')-10) +self.unpackDesn[2:4] +self.unpackDesn[5];                                                           
           if len(self.unpackDesn)<8 :                   # 00 -> 00
               self.packDesn = self.packDesn +'00'
           else:                                         # 162 -> G2
               num = self.unpackDesn[7:]
               if int(num) < 100:                            # 1 to 99 -> 01 to 99
                   self.packDesn = self.packDesn +num.zfill(2)       
               else:                                         # 162 -> G2
                   num = int(num[:2])            
                   if num>35:                    
                       num += 6  # ...XYZ[\]^_`abc...
                   self.packDesn = self.packDesn + chr(num+ord('A')-10) +self.unpackDesn[9]                                                                     
           self.packDesn = self.packDesn +self.unpackDesn[6] # S  
           return             
   # https://minorplanetcenter.net/Extended_Files/Extended_MPCORB_Data_Format_Manual.pdf        
   def unpackFlag(self, flag):     
       #object is PHA       
       if flag >= 32768:
           Ast.Tags += ' PHA '     
           flag -= 32768 
       #object is on critical list           
       if flag >= 16384:    
           flag -= 16384
       #object is 1-opp       
       if flag >= 8192:   
           flag -= 8192
       #object is 1km+NEO       
       if flag >= 4096:    
           flag -= 4096           
       #object is NEO       
       if flag >= 2048: 
           flag -= 2048           
       # unused and internal flags
       if flag >= 1024:
           flag -= 1024   
       if flag >= 512:
           flag -= 512            
       if flag >= 256:
           flag -= 256            
       if flag >= 128:
           flag -= 128            
       if flag >= 64:
           flag -= 64                   
       #object is (Scattered Disc??)
       if flag >= 17:
           Ast.Tags += ' TNO '     
           flag -= 17             
       #object is (Cubewano??)
       if flag >= 16:
           Ast.Tags += ' TNO '     
           flag -= 16  
       #object is (TNO??)
       if flag >= 15:
           Ast.Tags += ' TNO '     
           flag -= 15  
       #object is (Plutino??)
       if flag >= 14:
           Ast.Tags += ' TNO '     
           flag -= 14                                   
       #object is Distant Object (Centaur??)
       if flag >= 10:
           Ast.Tags += ' Centaur '     
           flag -= 10                      
       #object is Jupiter Trojan       
       if flag >= 9:    
           flag -= 9       
       #object is Hilda       
       if flag >= 8:  
           flag -= 8      
       #object is Phoecaea       
       if flag >= 7: 
           flag -= 7        
       #object is Hungaria       
       if flag >= 6: 
           flag -= 6        
       #object has q < 1.665 AU       
       if flag >= 5:    
           flag -= 5        
       #object is Amor       
       if flag >= 4:
           Ast.Tags += ' Amor '     
           flag -= 4        
       #object is Apollo       
       if flag >= 3:
           Ast.Tags += ' Apollo '     
           flag -= 3        
       #object is Aten       
       if flag >= 2:
           Ast.Tags += ' Aten '     
           flag -= 2        
       #object is Atira       
       if flag >= 1:   
           flag -= 1               
       
# process one .XML file for object designations 
import xml.etree.ElementTree
def get_xml_objects(f):
    objects = []
    root = xml.etree.ElementTree.parse(f).getroot()    
    for permID in root.iter('permID'):
        objects.append(permID.text)
    for provID in root.iter('provID'):
        objects.append(provID.text)     
    for trkSub in root.iter('trkSub'):
        objects.append(trkSub.text)           
    return objects

# process sesitek.txt to check object designation
def find_in_sesitek(desn):
    new_desn = desn
    f = open(os.path.join(directory, 'sesitek.txt'), "r")
    # for each line in sesitek
    for line in f:
        s = line.split()
        # skip blank line
        if len(s) == 0:
            continue
        # find line we need
        if s[0] == desn: 
            # write new unpacked designation
            new_desn = (line[19:]).strip()      
    return new_desn   
    
# find tag AsteroidList.Tags and return number of objects and observations 
def count_tag(AsteroidList, tag):
    n = 0 
    nobs = 0      
    for Ast in AsteroidList:
        if tag in Ast.Tags:
            n += 1
            nobs += Ast.NObs  
    line = str(n).rjust(6,' ') +'/' +str(nobs)        
    return line     

# merge Asteroid list duplicates
def merge_duplicates(AsteroidList):
    AsteroidList.sort(key=lambda x: x.packDesn) # sort the list of objects by .packDesn
    for i in range(len(AsteroidList)-1, 0, -1): # range(start, end, step)
        #search for duplicate packDesn
        if (AsteroidList[i].packDesn == AsteroidList[i-1].packDesn) and (AsteroidList[i].packDesn != ''):
            #take care of duplicate
            if AsteroidList[i-1].NEOCPDesn == '':
                AsteroidList[i-1].NEOCPDesn = AsteroidList[i].NEOCPDesn
            if AsteroidList[i].NEOCPDesn == '':
                AsteroidList[i].NEOCPDesn = AsteroidList[i-1].NEOCPDesn                
            AsteroidList[i-1].NObs += AsteroidList[i].NObs   
            print(AsteroidList[i].packDesn +' duplicate was deleted')    
            del AsteroidList[i]         
        #seach for duplicate NEOCPDesn
        if (AsteroidList[i].packDesn == '') and (AsteroidList[i].NEOCPDesn == AsteroidList[i-1].NEOCPDesn):
            #take care of duplicate           
            AsteroidList[i-1].NObs += AsteroidList[i].NObs 
            print(AsteroidList[i].NEOCPDesn +' duplicate was deleted')         
            del AsteroidList[i] 
    
################################################################################    

# search directory of this .py script for .xml files
from pathlib import Path
import os.path
import glob
directory = Path(__file__).parents[0]   
searchpath = os.path.join(directory, '*.xml') 
filenames = glob.glob(searchpath)
#print(filenames)

# get object designations from the .xml files
numfiles = 0
objectlist = []
for filename in filenames:
    # [1,2,3]+[4,5] = [1,2,3,[4,5]]
    #objectlist.append( get_xml_objects(filename))
    # [1,2,3]+[4,5] = [1,2,3,4,5]
    objectlist.extend( get_xml_objects(filename))
    numfiles += 1
#print(objectlist)    
print(str(numfiles) +' files were processed.')
print(str(len(objectlist)) +' observations were found.')

# count observations of each object
objectlist.sort()
object_counts = {} # dictionary {desn: count} pairs
for desn in objectlist:
    object_counts[desn] = objectlist.count(desn) # add item   
#print(object_counts)

# feed the dictionary into List of Asteroid class objects
# Asteroid class object = [packDesn, unpackDesn, NEOCPDesn, NObs, Found, Comet, Tags]
# AsteroidList = [Asteroid, Asteroid, Asteroid.. ]
# zmena udaju: AsteroidList[1].NObs = 5 
# zavolani procedury: AsteroidList[1].pack()
AsteroidList = [ Asteroid('', key, '', object_counts[key], False, False, '') for key in object_counts]
print('Total Objects %d' % Asteroid.objectCount)    
        
# compute packed designation of asteroids    
for Ast in AsteroidList:
    Ast.pack()

# check sesitek for NEOCPs
# while loop abychom zmeneny zaznam sjeli znova v dalsi iteraci,
# protoze nektera telesa jsou preznacovana opakovane
i = 0
while i < len(AsteroidList):  
    Ast = AsteroidList[i] 
    if (Ast.unpackDesn == '') and (Ast.NEOCPDesn != ''): 
        Ast.unpackDesn = find_in_sesitek(Ast.NEOCPDesn) # returns original or newer designation 
        if Ast.NEOCPDesn != Ast.unpackDesn: #designation changed
            print('NEOCP found in sesitek: ' +str(Ast.NEOCPDesn) +' = ' +str(Ast.unpackDesn))    
            i -= 1 # repeat loop for this object                                 
        Ast.pack() # neocp presune z unpackDesn do neocpDesn          
    i += 1      
                   
# check for duplicates
merge_duplicates(AsteroidList)
print('Objects in list: ' +str(len(AsteroidList)))     
   
# connect MySQL database
# http://zetcode.com/python/pymysql/
unknown = 0
unknownobs = 0
comets = 0
cometsobs = 0
asteroids = 0
asteroidsobs = 0
import pymysql
con = pymysql.connect(host=host,user=user,passwd=password,db=database) 
with con:  
    cur = con.cursor()   
    
    ###----------------------------------------------------------###
    #  Part 1: Get newer designations for Asteroids from mpcorb    #
    ###----------------------------------------------------------###  
       
    i = 0
    while i < len(AsteroidList):        
        Ast = AsteroidList[i] 
        i += 1       
            
        # skip comets and neocps
        if ((Ast.Comet) or (Ast.packDesn == '')):
            continue               
                        
        # asteroids
        else:                  
            # try to find in mpcorb.desn  
            query = "SELECT flags FROM mpcorb WHERE desn='" +Ast.packDesn +"'"        
            cur.execute(query)     
            # if not found, try to find out newer designation (asteroid was numbered in meanwhile)            
            if cur.rowcount == 0:                
                cur.execute("SELECT desn, design FROM mpcorb WHERE design like '%" +Ast.unpackDesn +"'")                   
                
                if cur.rowcount > 0:           
                    resline = cur.fetchone()     # [desn, design]
                    if resline[1][0] == '(':     # (12954) 2040 T-2
                    # newer designation found: save old one into Ast.NEOCPDesn and load up the new designation
                        note = 'Newer designation found in mpcorb. '+str(Ast.unpackDesn)
                        design = resline[1].split()  # [(12954), 2040,T-2]                                  
                        Ast.NEOCPDesn = Ast.unpackDesn
                        Ast.unpackDesn = design[0][1:-1] #'12954' from '(12954)' 
                        note = note +' = ' +str(Ast.unpackDesn)
                        Ast.pack()
                        print(note)  
                        # adjust counter to repeat search with new designation
                        i -= 1                       
    
    # check for duplicates (muze zmenit poradi, pouzivat vne indexu!)
    merge_duplicates(AsteroidList)  
            
    ###-----------------------------------------------###
    #  Part 2: Count and get asteroid tags from MPCORB  #
    ###-----------------------------------------------###    
 
    for Ast in AsteroidList:      
            
        # comets
        if Ast.Comet:
            comets += 1
            cometsobs += Ast.NObs
            Ast.Found = True
            Ast.displayAsteroid()    
        
        # neocps
        elif Ast.packDesn == '':
            print("NEOCP Object {} was not found." .format(Ast.NEOCPDesn))            
            unknown += 1
            unknownobs += Ast.NObs
            Ast.displayAsteroid()              
                  
        # asteroids
        else:                             
            #print("The query affected {} rows".format(cur.rowcount))  
            cur.execute("SELECT flags, incl, e, a FROM mpcorb WHERE desn='" +Ast.packDesn +"'")             
            
            # found: process the result
            if cur.rowcount > 0:
                resline = cur.fetchone()   
                # PHA, NEA, Amor, Aten, Apollo ....           
                Ast.unpackFlag(resline[0])     
                # Mars Crosser (perihelion 1.3-1.67) 
                # (musi a > 1.0 ???)
                q = (float(resline[3]) * (1.0 - float(resline[2])))
                if (q > 1.3) and (q < 1.67):
                    Ast.Tags += ' MC '          
                # Outer Planets Crosser (peri. mensi a afel. vetsi nez vzdalenost planet)   
                # ((( a <= 5.0) || (a >= 5.4) || (e >= 0.3))
                q = (float(resline[3]) * (1.0 - float(resline[2])))    
                Q = (float(resline[3]) * (1.0 + float(resline[2])))    
                if ((q < 5.2) and (Q > 5.2)) or ((q < 9.6) and (Q > 9.6)) or ((q < 19.3) and (Q > 19.3)) or ((q < 30.3) and (Q > 30.3)): 
                    Ast.Tags += ' OPC '            
                # i > 35.0 
                if float(resline[1]) > 35:
                    Ast.Tags += ' I '               
                # e > 0.4   
                if float(resline[2]) > 0.4:
                    Ast.Tags += ' e '                  
                asteroids += 1
                asteroidsobs += Ast.NObs     
                Ast.Found = True                                       
                Ast.displayAsteroid()      
                
            # not found
            if cur.rowcount == 0:
                print("Object {} was not found in MPCORB." .format(Ast.packDesn))            
                unknown += 1
                unknownobs += Ast.NObs
                Ast.displayAsteroid() 
 
print('Objects in list: ' +str(len(AsteroidList)))
                
# generate text file
f= open(os.path.join(directory, 'puc.txt'),"w+")
f.write('-'*78 +"\n")
f.write('               Statistics of observations found in current directory' +"\n")
f.write('-'*78 +"\n")
f.write("\n")
f.write('Working directory: ' +str(directory) +"\n")
f.write("\n")
f.write('Processed # ' +str(numfiles) +' files with #' +str(len(objectlist)) +' observations' +"\n")
f.write("\n")
f.write('SUMMARY of observation types:' +"\n")
f.write(str(unknown).rjust(6,' ')   +'  UNKNOWN OBJECTS  in' +str(unknownobs).rjust(6,' ')   +' observations' +"\n")
f.write(str(comets).rjust(6,' ')    +'  COMETS           in' +str(cometsobs).rjust(6,' ')    +' observations' +"\n")
f.write(str(asteroids).rjust(6,' ') +'  MINOR PLANETS    in' +str(asteroidsobs).rjust(6,' ') +' observations' +"\n")
f.write('  --------------------------------------------' +"\n")
f.write(str(unknown+comets+asteroids).rjust(6,' ')   +'  OBJECTs          in' +str(unknownobs+cometsobs+asteroidsobs).rjust(6,' ')   +' observations' +"\n")
f.write("\n")
f.write('Details for MPs...(object/observation): ' +"\n")
f.write('   All.............' +str(asteroids).rjust(6,' ') +'/' +str(asteroidsobs) +"\n")
f.write('   Mars Crossers...' +count_tag(AsteroidList, ' MC ') +"\n")
f.write('   Amors...........' +count_tag(AsteroidList, ' Amor ') +"\n")
f.write('   Apollos.........' +count_tag(AsteroidList, ' Apollo ') +"\n")
f.write('   Atens...........' +count_tag(AsteroidList, ' Aten ') +"\n")
f.write('   Centaurs........' +count_tag(AsteroidList, ' Centaur ') +"\n")
f.write('   OPCs............' +count_tag(AsteroidList, ' OPC ') +"\n")
f.write('   TNOs............' +count_tag(AsteroidList, ' TNO ') +"\n")
f.write('of which:'+"\n")
f.write('    PHAs...........' +count_tag(AsteroidList, ' PHA ') +"\n")
f.write('    i > 35.0 ......' +count_tag(AsteroidList, ' I ') +"\n")
f.write('    e >  0.4 ......' +count_tag(AsteroidList, ' e ') +"\n")
f.write("\n")
# write out asteroids with Tags [Amor Aten Apollo MC e I PHA OPC Centaur ] 
f.write('List of Unusual Minor Planets:'"\n")
for Ast in AsteroidList:
    if Ast.Tags != '':
        line = str(Ast.packDesn).rjust(7,' ')
        if Ast.NEOCPDesn != '':
            line += ' ( ' +Ast.NEOCPDesn +') ->'
        else:
            line += '            ->'
        line += str(Ast.NObs).rjust(4,' ') +'[' +Ast.Tags +']' 
        f.write(' * ' +line +"\n")    
f.write("\n")
# write out list of comets
f.write('List of Comets: '+"\n")
for Ast in AsteroidList:
    if Ast.Comet:
        line = str(Ast.packDesn).rjust(8,' ')
        if Ast.NEOCPDesn != '':
            line += ' ( ' +Ast.NEOCPDesn +') ->'
        else:
            line += '            ->'
        line += str(Ast.NObs).rjust(4,' ') +' times' 
        f.write(' + ' +line +"\n")  
f.write("\n")
# write out list of unknown objects
f.write('List of Unknown objects: '+"\n")
for Ast in AsteroidList:
    if not Ast.Found:
        line = Ast.packDesn+' '+Ast.unpackDesn+' '+Ast.NEOCPDesn+' in '+str(Ast.NObs)+' observations' 
        f.write('   ' +line +"\n") 
f.write("\n")
f.write('-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= E N D =-=-=-=-=-=-=-==-=-=-=-=-=-=-=-=-=-=-'+"\n")
f.write("\n")
f.write(" Full object table:"+"\n")
f.write("\n")
f.write(' packDesn  unpackDesn  NEOCPDesn NObs Comet? Tags'+"\n")
for Ast in AsteroidList:
    f.write(' ' +Ast.packDesn.ljust(10) +Ast.unpackDesn.ljust(12) +Ast.NEOCPDesn.ljust(10) +str(Ast.NObs).ljust(5) +str(Ast.Comet).ljust(7) +Ast.Tags+"\n")
f.write("------------------------------------------------------------------"+"\n")
# real sum of observations
n = 0
for Ast in AsteroidList:   
    n += Ast.NObs
f.write( str(len(AsteroidList)) +' objects in ' +str(n) +' observations.'+"\n")   
f.close
